package easik.states;

import java.util.Hashtable;
import java.util.LinkedList;
import java.util.Map;

import javax.swing.BorderFactory;


import org.jgraph.graph.AttributeMap;
import org.jgraph.graph.DefaultEdge;
import org.jgraph.graph.DefaultGraphCell;
import org.jgraph.graph.GraphConstants;

import easik.Easik;
import easik.sketch.edge.GuideEdge;
import easik.sketch.edge.SketchEdge;
import easik.sketch.path.SketchPath;
import easik.sketch.vertex.EntityNode;

/** 
 * The GetPathState is a state which can be used to retrieve a single path from the 
 * user. It ensures that the only edges which are selectable are those which are attached
 * to the previous edge.
 *
 * @author Rob Fletcher 2005
 * @author Kevin Green 2006
 * @author Vera Ranieri 2006
 * @version 2006-07-13 Kevin Green
 */
public class GetPathState extends EasikState {

	/**
	 * Contains elements of the sketch that are selectable
	 */
	private Map _selectable;
	/**
	 * Contains elements of the sketch that are not selectable
	 */
	private Map _unselectable;
	/**
	 * The path created from the edges chosen
	 */
	private SketchPath _pathToReturn;	
	/**
	 * Stores whether the next state is to add a new path
	 */
	private boolean _nextState;
	/**
	 * Stores whether the next state is the finish state
	 */
	private boolean _finishState;

	/**
	 * Default Constructor
	 * 
	 * @param next Boolean determining whether the user shoulbe be allowed to select next on this round.
	 *   True if next could be selected, false otherwise.
	 * @param finish Boolean determining whether the user should be allowed to select finish on this round.  
	 *   True if finish could be selected, false otherwise.
	 */
	public GetPathState(boolean next, boolean finish) {
		_pathToReturn = null;
		_nextState = next;
		_finishState = finish;
	}

	/**
	 * When pushed on, the first thing done is clearing the selection
	 * and then disabling selection for all items except for edges.
	 */
	public void pushedOn() {
		Easik.getInstance().getFrame().setStateString("Select a path: ");

		setNextButton(false);
		setCancelButton(true);
		setFinishButton(false);
		
		_selectable = new Hashtable();
		_unselectable = new Hashtable();

		GraphConstants.setSelectable(_selectable, true);
		GraphConstants.setSelectable(_unselectable, false);

		// Initially, we allow all edges and entities to be selected
		Easik.getInstance().getFrame().getSketch().clearSelection();
	}

	/**
	 * Update the selection so that the only selectable items will be those within
	 * reach of the existing edges.
	 */
	public void selectionUpdated() {
		Object[] entireSet = Easik.getInstance().getFrame().getSketch().getRoots();
		
		resetColors();
		
		Object[] newSelection = Easik.getInstance().getFrame().getSketch().getSelectionCells();
		for(int i=0; i<newSelection.length; i++){
			AttributeMap myMap = ((DefaultGraphCell)newSelection[i]).getAttributes();
			GraphConstants.setBorder(myMap, BorderFactory.createLineBorder(Easik.getInstance().getIni().getPATH_SELECTION_COLOR(), 1));
			GraphConstants.setForeground(myMap, Easik.getInstance().getIni().getPATH_SELECTION_COLOR());
			GraphConstants.setLineColor(myMap, Easik.getInstance().getIni().getPATH_SELECTION_COLOR());
			GraphConstants.setLineWidth(myMap, 2);
			((DefaultGraphCell)newSelection[i]).setAttributes(myMap);
		}

		// First check to see if the selection is empty
		if (newSelection.length == 0) {
			// If it is, then any edge is selectable
			Easik.getInstance().getFrame().getSketch().getGraphLayoutCache()
			.edit(Easik.getInstance().getFrame().getSketch().getRoots(), _selectable);
			
			LinkedList<Object> unselectables = new LinkedList<Object>();
			// This just loops through the entire set, adding the unselectable
			// cells. In this case it will be edges with the wrong
			// source, and those visual-helper edges.
			for (int i = 0; i < entireSet.length; i++) {
				// If it is not an edge... it is unselectable
				if (!(entireSet[i] instanceof DefaultEdge) && !(((DefaultGraphCell)entireSet[i]).getUserObject() instanceof EntityNode)) {
					unselectables.add(entireSet[i]);
				} 
				//If it is an Edge, but a visual guide edge... it is unselectable
				else if ((entireSet[i] instanceof DefaultEdge) &&
						(Easik.getInstance().getFrame().getSketch().getAdapter()
						.getEdgeFromCell((DefaultEdge) entireSet[i]) instanceof GuideEdge)) {
					unselectables.add(entireSet[i]);
				} 
			}
			
			Easik.getInstance().getFrame().getSketch().getGraphLayoutCache()
			.edit(unselectables.toArray(), _unselectable);
			
			// And we can't have a next/finish button available
			setNextButton(false);
			setFinishButton(false);
		} 
		else {
			if(((DefaultGraphCell)newSelection[0]).getUserObject() instanceof EntityNode){
				Easik.getInstance().getFrame().getSketch().getGraphLayoutCache()
				.edit(Easik.getInstance().getFrame().getSketch().getRoots(), _unselectable);
				
				Easik.getInstance().getFrame().getSketch().getGraphLayoutCache()
				.edit(newSelection, _selectable);
			}
			else{
				//If edges are selected, we'll grab the last edge
				DefaultEdge lastEdge =
					(DefaultEdge) newSelection[newSelection.length - 1];

				// Grab it's target
				Object goodTarget = lastEdge.getTarget();
				// We'll set up the selection to only allow those which are currently
				// selected, or any edge which has a source the same as the target
				resetSelection();
				LinkedList<Object> unselectables = new LinkedList<Object>();
				// This just loops through the entire set, adding the unselectable
				// cells. In this case it will be non-edges, edges with the wrong
				// source, and those visual-helper edges.
				for (int i = 0; i < entireSet.length; i++) {
					// If it is not an edge... it is unselectable
					if (!(entireSet[i] instanceof DefaultEdge)) {
						unselectables.add(entireSet[i]);
					} 
					//If it is an Edge, but a visual guide edge... it is unselectable
					else if ((Easik.getInstance().getFrame().getSketch().getAdapter()
							.getEdgeFromCell((DefaultEdge) entireSet[i]) instanceof GuideEdge)) {
						unselectables.add(entireSet[i]);
					} 
					else {
						// If the source of the edge is not the good target, then it isn't selectable
						if (((DefaultEdge) entireSet[i]).getSource()
							!= goodTarget) {
							unselectables.add(entireSet[i]);
						}
					}
				}

				// Mark the unselectable as unselectable, and the previous selection as selectable
				Easik.getInstance().getFrame().getSketch().getGraphLayoutCache()
					.edit(unselectables.toArray(),_unselectable);
				Easik.getInstance().getFrame().getSketch().getGraphLayoutCache()
					.edit(newSelection,	_selectable);
			}
			//Since we have something selected, then we can enable the 'next' button, if it should be.
			setNextButton(_nextState);
			
			//If the user can finish this round, then the Finish button should be activated.
			setFinishButton(_finishState);
		}
	}

	/**
	 * When the state gets popped, then it should tell the new top item what path it had
	 * collected before being popped.
	 * Since this is called AFTER popping, it can use peek() to get the top item.	 
	 */
	public void poppedOff() {
		((PathAcceptingState)Easik.getInstance().getStateManager().peekState())
			.passPath(_pathToReturn);
		
		resetColors();
		
		Easik.getInstance().getFrame().getSketch().getGraphLayoutCache().reload();
		Easik.getInstance().getFrame().getSketch().clearSelection();
	}

	/**
	 * If path collection has been cancelled, then pop off, and set the 
	 * path to be null.
	 */
	public void cancelClicked() {
		_pathToReturn = null;
		resetSelection();
		Easik.getInstance().getStateManager().popState();
	}

	/**
	 * When next is clicked, pop off after preparing an array containing the 
	 * edges in the path. Convert to the proper graph edge of the sketch.
	 */
	public void nextClicked() {
		resetSelection();
		LinkedList<SketchEdge> path = new LinkedList<SketchEdge>();
		int selLength = Easik.getInstance().getFrame().getSketch().getSelectionCells().length;
		if(selLength == 1 && ((DefaultGraphCell) Easik.getInstance().getFrame().getSketch().getSelectionCells()[0]).getUserObject() instanceof EntityNode){
			_pathToReturn = new SketchPath((EntityNode)((DefaultGraphCell)Easik.getInstance().getFrame().getSketch().getSelectionCells()[0]).getUserObject());
		}
		else{
			for (int i = 0; i < selLength; i++) {
				path.add((SketchEdge)((DefaultEdge)Easik.getInstance().getFrame().getSketch().getSelectionCells()[i]).getUserObject());
			}
			_pathToReturn =	new SketchPath(path);
		}
		Easik.getInstance().getStateManager().popState();
	}
	
	/**
	 * When finish is clicked, the stack is popped off after an array 
	 * containing the path is created.
	 */
	public void finishClicked(){
		nextClicked();
	}

	/**
	 * State string identifier.
	 * 
	 * @return String literal "Select a path"
	 */
	public String toString() {
		return "Select a path";
	}
	
	/**
	 * Set everything to be selectable
	 */
	private void resetSelection() {
		Easik.getInstance().getFrame().getSketch().getGraphLayoutCache().edit(
			Easik.getInstance().getFrame().getSketch().getRoots(),
			_selectable);
	}
}
